/**
 *-----------------------------------------------------------------------------------
 *    Filename: ADFRender.c
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Copyright 2004-2007 Mitsubishi Electric Research Laboratories (MERL)
 *    An implementation for ADF rendering setup computations and ADFRenderGlyph()
 *    Ronald Perry and Sarah Frisken
 *-----------------------------------------------------------------------------------
 */

/**
 *-----------------------------------------------------------------------------------
 *    START: iType Edge Rendering
 *-----------------------------------------------------------------------------------
 */
#include "fs_itype.h"
#ifdef FS_EDGE_RENDER

/**
 *-----------------------------------------------------------------------------------
 *    Required include files for this implementation
 *-----------------------------------------------------------------------------------
 */
/*#include <math.h>*/  /* not currently needed */
#include "adftypesystem.h"
#include "adfgenerate.h"
#include "adfimplicit.h"
#include "adffixedmath.h"



/**
 *-----------------------------------------------------------------------------------
 *    RENDERING SETUP COMPUTATIONS
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    CacheGlyphData is data determined from ADFRenderAttrs by ADFRenderSetup() to
 *    enable the application to position and typeset a cached density image of an ADF
 *    glyph via ADFPositionCachedImage(). This data structure defines the memory layout
 *    of the private API data structure ADFCacheGlyphData.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */

/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */


/**
 *-----------------------------------------------------------------------------------
 *    Setup for ADF glyph rendering. Application rendering attributes are processed
 *    by ADFRenderSetup() to determine:
 *
 *    a) ADFRenderGlyphData, such as the rendering transformation, required by 
 *       ADFRenderGlyph() to render the ADF glyph according to the application 
 *       rendering attributes.
 *
 *    b) ADFRenderImageAttrs, such as the image size and image origin, required by 
 *       the application to prepare a density image for rendering the ADF glyph.
 *
 *    c) ADFTypesetAttrs, required by the application for adjusting the font metrics 
 *       due to grid fitting.
 *
 *    d) ADFCacheGlyphData, required by the application for positioning and typesetting 
 *       a cached density image of the ADF glyph via ADFPositionCachedImage(). If this 
 *       data is not required (e.g., when density images are not being cached), set the 
 *       pointer to this structure to NULL.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    To build the rendering transformation required for rendering the ADF glyph, this
 *    function first computes the transformation from ADF coordinates to image
 *    coordinates, taking into account the specified rendering attributes (such as
 *    point size and dpi. Since implicit ADFs are
 *    being used, the transformation from ADF coordinates to image coordinates is
 *    simply copied into the rendering transformation (implicit ADF rendering requires
 *    an ADF to image transformation matrix).
 *
 *    When MAZ grid fitting is requested, (1) the image size and image origin computed
 *    by ADFRenderSetup() are identical to the image size and image origin obtained
 *    when grid fitting is disabled entirely and (2) the x and y coordinates of the
 *    requested pen position are rounded to the nearest integer values when determining
 *    the rendering transformation above, thereby ensuring that a glyph placed at two
 *    different pen positions will appear identical when rendered (see Usage Note:
 *    Rendering CJK Glyphs With Multiple Alignment Zones in ADFTypeSystem.h). Note that
 *    MAZ grid fitting is performed dynamically during rendering.
 *
 *    To determine the transformation from ADF coordinates to image coordinates when a
 *    non-zero rotation angle is specified, the following sequence of 3x3 component
 *    transformation matrices are concatenated and simplified:
 *
 *        M    = [A] [B] [C] [D] [E] where
 *        A    = 1 0 Rix, 0 1 Riy, 0 0 1
 *        B    = cosTheta -sinTheta 0, sinTheta cosTheta 0, 0 0 1
 *        C    = 1 0 -Rix, 0 1 -Riy, 0 0 1
 *        D    = sx 0 0, 0 sy 0, 0 0 1
 *        E    = 1 0 -Gx, 0 1 -Gy, 0 0 1
 *        Rix = x-coordinate of the rotation point in image coordinates (rotPtX - penX)
 *        Riy = y-coordinate of the rotation point in image coordinates (rotPtY - penY)
 *        Gx    = x-coordinate of the glyph origin in ADF coordinates
 *        Gy    = y-coordinate of the glyph origin in ADF coordinates
 *        sx    = ADFtoImgScaleX
 *        sy    = ADFtoImgScaleY
 *
 *        The final transformation matrix M is:
 *        M[0][0] = sx * cosTheta
 *        M[0][1] = -sy * sinTheta
 *        M[0][2] = Rix + [(-sx) * Gx - Rix] * cosTheta - [(-sy) * Gy - Riy] * sinTheta
 *        M[1][0] = sx * sinTheta
 *        M[1][1] = sy * cosTheta
 *        M[1][2] = Riy + [(-sy) * Gy - Riy] * cosTheta + [(-sx) * Gx - Rix] * sinTheta
 *        M[2][0] = 0
 *        M[2][1] = 0
 *        M[2][2] = 1
 *
 *        Note: To reflect the sampling convention described below, the glyph origin is
 *        shifted down and to the left by half a pixel (i.e., the x and y translation
 *        components of the matrix M are decreased by one half)
 *
 *    Note that the grid fitting of rotated glyphs is not supported (i.e., grid fitting
 *    is automatically disabled when a non-zero rotation angle is specified)
 *
 *    Note that any change to the computations performed by this function must be
 *    reflected in ADFPositionCachedImage() (i.e., the corresponding computations must
 *    be bit-exact).
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    The TrueType specification follows the convention that rasterization of glyphs is
 *    performed using samples located at pixel centers. However, rasterization can
 *    often be performed more efficiently using samples located at integer pixel
 *    coordinates. This library follows the TrueType convention while maintaining high
 *    efficiency by adjusting the rendering transformation so that during rasterization
 *    integer pixel coordinates correspond to distance samples located at pixel
 *    centers. To achieve this, the x and y translation components of the rendering
 *    transformation are decreased by half a pixel.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Note that the gridFittingOn Boolean (see below) is set to true if and only if (1)
 *    rotation is disabled and (2) the gridFitType element of renderAttrs is set to
 *    either ADF_GRID_FIT_PIXEL or ADF_GRID_FIT_SUB_PIXEL. In contrast, the
 *    gridFittingOn Boolean is always set to false if the gridFitType element of
 *    renderAttrs is set to ADF_GRID_FIT_MAZ_PIXEL.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
ADF_Void ADFRenderSetup (void *ADF, ADFRenderAttrs *renderAttrs,
                         ADFRenderGlyphData *renderGlyphData,
                         ADFRenderImageAttrs *renderImageAttrs,
                         ADFTypesetAttrs *typesetAttrs)
{
    {
        /**
         *----Fixed point math implementation
         */
        ADF_U32 vertStriping;
        ADF_U32 horzStriping;
        ADF_I1616 tx, ty;
        ADF_I1616 adfToImg[2][2];
        ADF_I1616 sx, sy;
        ADF_I1616 ADFtoImgScaleX;
        ADF_I1616 ADFtoImgScaleY;
        ADF_I1616 rotationAngle;
        ADF_I1616 penX;
        ADF_I1616 penY;


        /**
         *----Cast the opaque ADF pointer as an ADFGlyph
         */
        ADFGlyph *adf = (ADFGlyph *) ADF;


        /**
         *----Check for a NULL ADF
         */
        if (adf == 0) {
            renderImageAttrs->imageW = 0;
            renderImageAttrs->imageH = 0;
            return;
        }


        /**
         *----Set the vertical striping Boolean
         */
        vertStriping = (renderAttrs->displayMode == ADF_REND_MODE_RGBv) ||
        (renderAttrs->displayMode == ADF_REND_MODE_BGRv);


        /**
         *----Set the horizontal striping Boolean
         */
        horzStriping = (renderAttrs->displayMode == ADF_REND_MODE_RGBh) ||
        (renderAttrs->displayMode == ADF_REND_MODE_BGRh);


        /**
         *----Convert the specified rotation angle to an ADF_I1616 fixed point value
         */
        rotationAngle = FLOAT_TO_I1616(renderAttrs->rotationAngle);


        /**
         *----Compute the unadjusted scale factors from ADF coordinates to pixel
         *----coordinates (ADFtoImgScaleX, ADFtoImgScaleY) using the point size,
         *----display resolution, EM box size in ADF coordinates, and additional
         *----scale factors specified in the rendering attributes. Note that a
         *----"point" is a physical distance, equaling 1/72th of an inch, and ppem is
         *----the number of pixels per EM (in x and y). 
         */
        {
            ADF_I32 divStatus;
            FS_CONST ADF_I1616 maxScale2048 = 0x8000000;
            FS_CONST ADF_I1616 inv72 = 0x000038e;
            ADF_I1616 ppem = I1616_MUL(I1616_MUL(FLOAT_TO_I1616(
            renderAttrs->pointSize), inv72), U32_TO_I1616(renderAttrs->dpi));
            ADF_I1616 ppemX = I1616_MUL(ppem, FLOAT_TO_I1616(renderAttrs->scaleX));
            ADF_I1616 ppemY = I1616_MUL(ppem, FLOAT_TO_I1616(renderAttrs->scaleY));
            ADF_I1616 invADFUnitsPerEM = I1616_DIV(I1616_CONST_1,
            FLOAT_TO_I1616(adf->ADFUnitsPerEM), &divStatus);


            /**
             *----Clamp ppemX and ppemY to the range [0,2048] to prevent overflows
             *----from occurring while processing implicit ADFs using ADF_I1616 fixed
             *----point arithmetic
             */
            if (ppemX < 0) ppemX = 0;
            else if (ppemX > maxScale2048) ppemX = maxScale2048;
            if (ppemY < 0) ppemY = 0;
            else if (ppemY > maxScale2048) ppemY = maxScale2048;
            ADFtoImgScaleX = I1616_MUL(ppemX, invADFUnitsPerEM);
            ADFtoImgScaleY = I1616_MUL(ppemY, invADFUnitsPerEM);
        }


        /**
         *----Initialize the adjusted scale factors (sx, sy) to the unadjusted scale
         *----factors (ADFtoImgScaleX, ADFtoImgScaleY)
         */
        sx = ADFtoImgScaleX;
        sy = ADFtoImgScaleY;


        /**
         *----Convert the specified pen position from floating point screen
         *----coordinates to ADF_I1616 fixed point screen coordinates
         */
        penX = FLOAT_TO_I1616(renderAttrs->penX);
        penY = FLOAT_TO_I1616(renderAttrs->penY);


        /**
         *----Determine if rotation is specified
         */
        if (rotationAngle) {



            /**
             *----Determine the ADF-to-image transformation
             */
            if (sx && sy) {


                /**
                 *----Both adjusted scale factors are non-zero, thus a non-trivial
                 *----ADF-to-image transformation can be computed
                 */
                ADF_I1616 sinTheta = I1616_SIN(rotationAngle);
                ADF_I1616 cosTheta = I1616_COS(rotationAngle);
                ADF_I1616 rx = FLOAT_TO_I1616(renderAttrs->rotationPtX);
                ADF_I1616 ry = FLOAT_TO_I1616(renderAttrs->rotationPtY);


                /**
                 *----Initialize the ADF-to-image transformation for scaling and
                 *----rotating the glyph (the translation vector is determined below)
                 */
                adfToImg[0][0] = I1616_MUL( cosTheta, sx);
                adfToImg[0][1] = I1616_MUL(-sinTheta, sy);
                adfToImg[1][0] = I1616_MUL( sinTheta, sx);
                adfToImg[1][1] = I1616_MUL( cosTheta, sy);


                /**
                 *----Convert the rotation point to image coordinates
                 */
                rx -= penX;
                ry -= penY;


                /**
                 *----Set the translation vector
                 */
                {
                    ADF_I1616 sxGx = I1616_MUL(-sx,
                    FLOAT_TO_I1616(adf->glyphOriginX));
                    ADF_I1616 syGy = I1616_MUL(-sy,
                    FLOAT_TO_I1616(adf->glyphOriginY));
                    tx = rx + I1616_MUL(sxGx - rx, cosTheta) -
                    I1616_MUL(syGy - ry, sinTheta);
                    ty = ry + I1616_MUL(syGy - ry, cosTheta) +
                    I1616_MUL(sxGx - rx, sinTheta);
                }


            } else {


                /**
                 *----At least one of the adjusted scale factors is zero resulting in
                 *----a trivial (i.e., a zero) ADF-to-image transformation
                 */
                tx = 0;
                ty = 0;
                adfToImg[0][0] = 0;
                adfToImg[0][1] = 0;
                adfToImg[1][0] = 0;
                adfToImg[1][1] = 0;
            }


            /**
             *----Translate the glyph origin so that it lies at the pen position
             */
            tx += penX;
            ty += penY;


            /**
             *----For efficiency, both implicit and explicit ADFs are sampled at
             *----integer pixel coordinates during rendering. However, by convention,
             *----TTF glyphs are rasterized by sampling at pixel centers (e.g., at
             *----(i+0.5, j+0.5) for CRT rendering, where i and j are integer pixel
             *----coordinates). To satisfy this convention without reducing ADF
             *----rendering efficiency, shift the glyph origin down and to the left
             *----by half a pixel.
             */
            tx -= I1616_CONST_ONE_HALF;
            ty -= I1616_CONST_ONE_HALF;


            /**
             *----Because grid fitting is automatically disabled when rotating, the
             *----incremental pen position adjustments are both zero
             */
            typesetAttrs->gridAlgnAdjX = 0;
            typesetAttrs->gridAlgnAdjY = 0;


        } else {


            /**
             *----Translate the glyph origin to the ADF coordinate space origin and
             *----scale by the adjusted scale factors
             */
            adfToImg[0][0] = sx;
            adfToImg[0][1] = 0;
            adfToImg[1][0] = 0;
            adfToImg[1][1] = sy;
            tx = I1616_MUL(-sx, FLOAT_TO_I1616(adf->glyphOriginX));
            ty = I1616_MUL(-sy, FLOAT_TO_I1616(adf->glyphOriginY));


            /**
             *----Translate the glyph origin so that it lies at the pen position
             */
            tx += penX;
            ty += penY;


            /**
             *----For efficiency, both implicit and explicit ADFs are sampled at
             *----integer pixel coordinates during rendering. However, by convention,
             *----TTF glyphs are rasterized by sampling at pixel centers (e.g., at
             *----(i+0.5, j+0.5) for CRT rendering, where i and j are integer pixel
             *----coordinates). To satisfy this convention without reducing ADF
             *----rendering efficiency, shift the glyph origin down and to the left
             *----by half a pixel.
             */
            tx -= I1616_CONST_ONE_HALF;
            ty -= I1616_CONST_ONE_HALF;


            /**
             *----Initialize the incremental pen position adjustment to (0,0)
             */
            typesetAttrs->gridAlgnAdjX = 0;
            typesetAttrs->gridAlgnAdjY = 0;

        }

        /**
         *----Determine the axis-aligned bounding box of the image and translate the
         *----glyph so that the bottom-left corner of the image's bounding box is at
         *----(0,0) in image coordinates
         */
        {
            ADF_U16 w, h;
            ADF_I1616 ox, oy;


            /**
             *----Compute the x limits of the bounding box of the image from the x
             *----limits of the transformed [0.0,1.0] x [0.0,1.0] ADF coordinate
             *----system
             */
            ADF_I1616 p1 = adfToImg[0][0];
            ADF_I1616 p2 = adfToImg[0][1];
            ADF_I1616 p3 = p1 + p2;
            ADF_I1616 min = ADF_MIN(0, ADF_MIN(p1, ADF_MIN(p2, p3)));
            ADF_I1616 max = ADF_MAX(0, ADF_MAX(p1, ADF_MAX(p2, p3)));


            /**
             *----Determine the image width and the x-coordinate of the image origin
             *----based on the striping of the display
             */
            if (!vertStriping) {


                /**
                 *----No striping in the vertical direction. The intensity of a pixel
                 *----is determined by sampling the ADF distance at the center of the
                 *----pixel. The width of the image required to entirely contain the
                 *----rendered glyph depends on the projected size of the ADF
                 *----bounding box (i.e., max - min) and the placement of the
                 *----projected ADF bounding box on the pixel grid, which together
                 *----can result in an image width of up to one additional pixel.
                 *----Compute the image width required to render the entire glyph for
                 *----any placement of the ADF on the pixel grid.
                 */
                if (min < max) w = (ADF_U16) I1616_TO_I32(max - min + I1616_CONST_1);
                else w = 0;


                /**
                 *----Determine the x-coordinate of the image origin by rounding the
                 *----projected ADF origin to the nearest pixel boundary (in x) and
                 *----translate the image (in x) to (0,0) in image coordinates
                 */
                ox = I1616_FLOOR(min + tx + I1616_CONST_ONE_HALF);
                tx -= ox;


            } else {


                /**
                 *----Striping in the vertical direction. The intensity of the three
                 *----vertical pixel components are determined by sampling the ADF
                 *----distance at the pixel center and at 1/3 of a pixel to the left
                 *----and to the right of the pixel center. The width of the image
                 *----required to entirely contain the rendered glyph depends on the
                 *----projected size of the ADF bounding box (i.e., max - min) and
                 *----the placement of the projected ADF bounding box on the
                 *----sub-pixel grid, which together can result in an image width of
                 *----up to 5/3 of an additional pixel. Compute the image width
                 *----required to render the entire glyph for any placement of the
                 *----ADF on the sub-pixel grid.
                 */
                if (min < max) w = (ADF_U16) I1616_TO_I32(max - min +
                I1616_CONST_FIVE_THIRDS);
                else w = 0;


                /**
                 *----Determine the x-coordinate of the image origin by rounding the
                 *----projected ADF origin to the nearest sub-pixel boundary (in x)
                 *----and translate the image (in x) to (0,0) in image coordinates
                 */
                ox = I1616_FLOOR(min + tx + I1616_CONST_ONE_SIXTH);
                tx -= ox;
            }


            /**
             *----Compute the y limits of the bounding box of the image from the y
             *----limits of the transformed [0.0,1.0] x [0.0,1.0] ADF coordinate
             *----system
             */
            p1 = adfToImg[1][0];
            p2 = adfToImg[1][1];
            p3 = p1 + p2;
            min = ADF_MIN(0, ADF_MIN(p1, ADF_MIN(p2, p3)));
            max = ADF_MAX(0, ADF_MAX(p1, ADF_MAX(p2, p3)));


            /**
             *----Determine the image height and the y-coordinate of the image origin
             *----based on the striping of the display
             */
            if (!horzStriping) {


                /**
                 *----No striping in the horizontal direction. The intensity of a
                 *----pixel is determined by sampling the ADF distance at the center
                 *----of the pixel. The height of the image required to entirely
                 *----contain the rendered glyph depends on the projected size of the
                 *----ADF bounding box (i.e., max - min) and the placement of the
                 *----projected ADF bounding box on the pixel grid, which together
                 *----can result in an image height of up to one additional pixel.
                 *----Compute the image height required to render the entire glyph
                 *----for any placement of the ADF on the pixel grid.
                 */
                if (min < max) h = (ADF_U16) I1616_TO_I32(max - min + I1616_CONST_1);
                else h = 0;


                /**
                 *----Determine the y-coordinate of the image origin by rounding the
                 *----projected ADF origin to the nearest pixel boundary (in y) and
                 *----translate the image (in y) to (0,0) in image coordinates
                 */
                oy = I1616_FLOOR(min + ty + I1616_CONST_ONE_HALF);
                ty -= oy;


            } else {


                /**
                 *----Striping in the horizontal direction. The intensity of the
                 *----three horizontal pixel components are determined by sampling
                 *----the ADF distance at the pixel center and at 1/3 of a pixel to
                 *----the top and to the bottom of the pixel center. The height of
                 *----the image required to entirely contain the rendered glyph
                 *----depends on the projected size of the ADF bounding box (i.e.,
                 *----max - min) and the placement of the projected ADF bounding box
                 *----on the sub-pixel grid, which together can result in an image
                 *----height of up to 5/3 of an additional pixel. Compute the image
                 *----height required to render the entire glyph for any placement of
                 *----the ADF on the sub-pixel grid.
                 */
                if (min < max) h = (ADF_U16) I1616_TO_I32(max - min +
                I1616_CONST_FIVE_THIRDS);
                else h = 0;


                /**
                 *----Determine the y-coordinate of the image origin by rounding the
                 *----projected ADF origin to the nearest sub-pixel boundary (in y)
                 *----and translate the image (in y) to (0,0) in image coordinates
                 */
                oy = I1616_FLOOR(min + ty + I1616_CONST_ONE_SIXTH);
                ty -= oy;
            }


            /**
             *----Record the size and the origin of the image
             */
            renderImageAttrs->imageW = w;
            renderImageAttrs->imageH = h;
            renderImageAttrs->imageOriginX = I1616_TO_I32(ox);
            renderImageAttrs->imageOriginY = I1616_TO_I32(oy) - 1;
        }


        /**
         *----Set the rendering transformation based on the ADF type selection
         */
#if (ADF_USE_IMPLICIT_ADFS)
        {
            /**
             *----Force integer (penX,penY) coordinates when using MAZ alignment
             *----zones and rotation is disabled. This ensures identical alignment
             *----behavior for any pen position.
             */
            if ((renderAttrs->gridFitType == ADF_GRID_FIT_MAZ_PIXEL) &&
                (rotationAngle == 0)) 
            {
                tx -= penX;
                tx += I1616_FLOOR(penX + I1616_CONST_ONE_HALF);
                ty -= penY;
                ty += I1616_FLOOR(penY + I1616_CONST_ONE_HALF);
            }


            /**
             *----Copy the ADF to image transformation into the rendering
             *----transformation
             */
            renderGlyphData->xform[0][0] = I1616_TO_FLOAT(adfToImg[0][0]);
            renderGlyphData->xform[0][1] = I1616_TO_FLOAT(adfToImg[0][1]);
            renderGlyphData->xform[0][2] = I1616_TO_FLOAT(tx);
            renderGlyphData->xform[1][0] = I1616_TO_FLOAT(adfToImg[1][0]);
            renderGlyphData->xform[1][1] = I1616_TO_FLOAT(adfToImg[1][1]);
            renderGlyphData->xform[1][2] = I1616_TO_FLOAT(ty);
            renderGlyphData->xform[2][0] = 0 /*0.0f*/;
            renderGlyphData->xform[2][1] = 0 /*0.0f*/;
            renderGlyphData->xform[2][2] = 65536 /*1.0f*/;
        }
#else
#error "Fixed point arithmetic is not supported for explicit ADFs."
#endif


        /**
         *----Set the filter cutoff values and filterCutoffScale based on the ADF
         *----type selection
         */
#if (ADF_USE_IMPLICIT_ADFS)
        {
            /**
             *----Initialize the updateRequired Boolean to false and convert the
             *----user-specified cutoff values from floating point pixel coordinates
             *----to ADF_I1616 fixed point pixel coordinates
             */
            ADF_I32      updateRequired = 0;
            ADF_I1616 ic = FLOAT_TO_I1616(renderAttrs->insideCutoff);
            ADF_I1616 oc = FLOAT_TO_I1616(renderAttrs->outsideCutoff);


            /**
             *----If the inside cutoff value is less than the outside cutoff value,
             *----swap the two cutoff values and set the updateRequired Boolean to
             *----true
             */
            if (ic < oc) {
                ADF_I1616 tmp = ic;
                ic = oc;
                oc = tmp;
                updateRequired = 1;
            }


            /**
             *----Clamp the outside cutoff value to the range [-20,20]
             */
            if (oc < I1616_CONST_NEG_20) {
                oc = I1616_CONST_NEG_20;
                updateRequired = 1;
            } else if (oc > I1616_CONST_20) {
                oc = I1616_CONST_20;
                updateRequired = 1;
            }


            /**
             *----Clamp the inside cutoff value to the range [-20,20]
             */
            if (ic < I1616_CONST_NEG_20) {
                ic = I1616_CONST_NEG_20;
                updateRequired = 1;
            } else if (ic > I1616_CONST_20) {
                ic = I1616_CONST_20;
                updateRequired = 1;
            }


            /**
             *----If the updateRequired Boolean is true (i.e., a change was made to
             *----one of the user-specified cutoff values), convert the modified
             *----cutoff values from ADF_I1616 fixed point pixel coordinates to
             *----floating pint pixel coordinates and store the results into
             *----renderGlyphData. Otherwise, simply copy the input values (which are
             *----already in floating pint pixel coordinates) to renderGlyphData.
             */
            if (updateRequired) {
                renderGlyphData->insideCutoff = I1616_TO_FLOAT(ic);
                renderGlyphData->outsideCutoff = I1616_TO_FLOAT(oc);
            } else {
                renderGlyphData->insideCutoff = renderAttrs->insideCutoff;
                renderGlyphData->outsideCutoff = renderAttrs->outsideCutoff;
            }


            /**
             *----Set the filter cutoff scale to 1
             */
            renderGlyphData->filterCutoffScale = 65536 /*1.0f*/;
        }
#else 
#error "Fixed point arithmetic is not supported for explicit ADFs."
#endif


        /**
         *----Copy other rendering attributes
         */
        renderGlyphData->gamma = renderAttrs->gamma;
        renderGlyphData->displayMode = renderAttrs->displayMode;
        renderGlyphData->useColorReduction = renderAttrs->useColorReduction;
        renderGlyphData->colorReductionAmt = renderAttrs->colorReductionAmt;


        /**
         *----If a non-zero rotation angle is specified, set the gridFitType
         *----attribute in renderGlyphData to ADF_GRID_FIT_NONE. Otherwise, copy the
         *----gridFitType attribute from the ADFRenderAttrs data structure. This
         *----attribute is used by the implicit ADF rendering implementation (see
         *----ADFImplicitFloat.c and ADFImplicitFixed.c) to determine if MAZ
         *----alignment zone detection and grid fitting should be applied during
         *----rendering.
         */
        renderGlyphData->gridFitType = (rotationAngle != 0) ?
        (ADF_GRID_FIT_NONE) : (renderAttrs->gridFitType);


        /**
         *----Determine the number of pixels per EM
         */
        renderGlyphData->ppem =
        I1616_TO_FLOAT(I1616_MUL(I1616_MUL(FLOAT_TO_I1616(renderAttrs->pointSize),
        0x000038e), U32_TO_I1616(renderAttrs->dpi)));


        /**
         *----Determine the unadjusted and adjusted typesetting scale factors
         */
        {
            ADF_I1616 FUToADFScale = FLOAT_TO_I1616(adf->FUToADFScale);
            typesetAttrs->FUToPixelScaleX = I1616_TO_FLOAT(
            I1616_MUL(FUToADFScale, ADFtoImgScaleX));
            typesetAttrs->FUToPixelScaleY = I1616_TO_FLOAT(
            I1616_MUL(FUToADFScale, ADFtoImgScaleY));
            typesetAttrs->adjFUToPixelScaleX = I1616_TO_FLOAT(
            I1616_MUL(FUToADFScale, sx));
            typesetAttrs->adjFUToPixelScaleY = I1616_TO_FLOAT(
            I1616_MUL(FUToADFScale, sy));
        }

    }
}

/**
 *-----------------------------------------------------------------------------------
 *    DRIVER FOR ADF RENDERING
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Render the ADF glyph into the specified density image using the rendering data
 *    ADFRenderGlyphData determined by ADFRenderSetup()
 *-----------------------------------------------------------------------------------
 */
ADF_Void ADFRenderGlyph (void *libInst, void *ADF, ADFRenderGlyphData *renderGlyphData,
                         ADFImage *image)
{
    /**
     *----Render the given ADF
     */
    {
        /**
         *----Render the implicit ADF
         */
        ADFGlyph *adf = (ADFGlyph *) ADF;
        ADFRenderGlyphImplicit(libInst, adf, renderGlyphData, image);
        return;
    }
}

/**
 *-----------------------------------------------------------------------------------
 *    END: iType Edge Rendering
 *-----------------------------------------------------------------------------------
 */
#endif
